/* BEGIN software license
 *
 * msXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright (C) 2009--2020 Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This file is part of the msXpertSuite project.
 *
 * The msXpertSuite project is the successor of the massXpert project. This
 * project now includes various independent modules:
 *
 * - massXpert, model polymer chemistries and simulate mass spectrometric data;
 * - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */


/////////////////////// StdLib includes
#include <cmath>
#include <iostream>
#include <iomanip>


/////////////////////// Qt includes
#include <QWidget>
#include <QThread>
#include <QInputDialog>


/////////////////////// QCustomPlot


/////////////////////// libXpertMass includes
#include <MsXpS/libXpertMassCore/MassDataCborMassSpectrumHandler.hpp>
#include <MsXpS/libXpertMassCore/MassPeakShaperConfig.hpp>
#include <MsXpS/libXpertMassCore/MassPeakShaper.hpp>
#include <MsXpS/libXpertMassGui/ColorSelector.hpp>


/////////////////////// pappsomspp includes
#include <pappsomspp/gui/plotwidget/massspectraceplotwidget.h>
#include <pappsomspp/core/processing/combiners/tracepluscombiner.h>
#include <pappsomspp/core/processing/combiners/traceminuscombiner.h>
#include <pappsomspp/core/processing/combiners/massspectrumpluscombiner.h>
#include <pappsomspp/core/processing/combiners/massspectrumminuscombiner.h>
#include <pappsomspp/gui/plotwidget/massspectraceplotcontext.h>
#include <pappsomspp/core/processing/combiners/integrationscope.h>


/////////////////////// Local includes
#include "MassSpecTracePlotCompositeWidget.hpp"
#include "MassSpecTracePlotWnd.hpp"
#include "ProgramWindow.hpp"
#include "ui_BasePlotCompositeWidget.h"

namespace MsXpS
{
namespace MineXpert
{


MassSpecTracePlotCompositeWidget::MassSpecTracePlotCompositeWidget(
  QWidget *parent, const QString &x_axis_label, const QString &y_axis_label)
  : BaseTracePlotCompositeWidget(parent, x_axis_label, y_axis_label)
{
  // qDebug();

  setupWidget();
}

MassSpecTracePlotCompositeWidget::~MassSpecTracePlotCompositeWidget()
{
  // qDebug();
}

void
MassSpecTracePlotCompositeWidget::setupWidget()
{
  // qDebug();

  /************  The QCustomPlot widget *************/
  /************  The QCustomPlot widget *************/
  mp_plotWidget =
    new pappso::MassSpecTracePlotWidget(this, m_axisLabelX, m_axisLabelY);

  m_ui->qcpTracePlotWidgetHorizontalLayout->addWidget(mp_plotWidget);

  // We want that the axis labels be output inside the plot wiget area.
  mp_plotWidget->yAxis->setTickLabelSide(QCPAxis::LabelSide::lsInside);

  connect(
    mp_plotWidget, &pappso::MassSpecTracePlotWidget::setFocusSignal, [this]() {
      BaseTracePlotCompositeWidget::mp_parentWnd->plotWidgetGotFocus(this);
    });

  connect(mp_plotWidget,
          &pappso::MassSpecTracePlotWidget::lastCursorHoveredPointSignal,
          this,
          &MassSpecTracePlotCompositeWidget::lastCursorHoveredPoint);

  connect(mp_plotWidget,
          &pappso::MassSpecTracePlotWidget::plotRangesChangedSignal,
          mp_parentWnd,
          &BaseTracePlotWnd::plotRangesChanged);

  connect(mp_plotWidget,
          &pappso::MassSpecTracePlotWidget::xAxisMeasurementSignal,
          this,
          &MassSpecTracePlotCompositeWidget::xAxisMeasurement);

  // For this one we need to force the cast to the
  // pappso::MassSpecTracePlotWidget because the context that is passed as
  // argument is specific of that class.
  connect(static_cast<pappso::MassSpecTracePlotWidget *>(mp_plotWidget),
          static_cast<void (pappso::MassSpecTracePlotWidget::*)(
            const pappso::MassSpecTracePlotContext &)>(
            &pappso::MassSpecTracePlotWidget::keyPressEventSignal),
          this,
          static_cast<void (MassSpecTracePlotCompositeWidget::*)(
            const pappso::MassSpecTracePlotContext &)>(
            &MassSpecTracePlotCompositeWidget::plotWidgetKeyPressEvent));

  connect(mp_plotWidget,
          &pappso::MassSpecTracePlotWidget::keyReleaseEventSignal,
          this,
          &MassSpecTracePlotCompositeWidget::plotWidgetKeyReleaseEvent);

  connect(static_cast<pappso::MassSpecTracePlotWidget *>(mp_plotWidget),
          static_cast<void (pappso::MassSpecTracePlotWidget::*)(
            const pappso::MassSpecTracePlotContext &)>(
            &pappso::MassSpecTracePlotWidget::mousePressEventSignal),
          this,
          static_cast<void (MassSpecTracePlotCompositeWidget::*)(
            const pappso::MassSpecTracePlotContext &)>(
            &MassSpecTracePlotCompositeWidget::plotWidgetMousePressEvent));

  connect(static_cast<pappso::MassSpecTracePlotWidget *>(mp_plotWidget),
          static_cast<void (pappso::MassSpecTracePlotWidget::*)(
            const pappso::MassSpecTracePlotContext &)>(
            &pappso::MassSpecTracePlotWidget::mouseReleaseEventSignal),
          this,
          static_cast<void (MassSpecTracePlotCompositeWidget::*)(
            const pappso::MassSpecTracePlotContext &)>(
            &MassSpecTracePlotCompositeWidget::plotWidgetMouseReleaseEvent));

  connect(mp_plotWidget,
          &pappso::MassSpecTracePlotWidget::plottableSelectionChangedSignal,
          this,
          &BasePlotCompositeWidget::plottableSelectionChanged);

  connect(static_cast<pappso::MassSpecTracePlotWidget *>(mp_plotWidget),
          &pappso::MassSpecTracePlotWidget::massDeconvolutionSignal,
          this,
          &MassSpecTracePlotCompositeWidget::plotWidgetMassDeconvolution);

  connect(
    static_cast<pappso::MassSpecTracePlotWidget *>(mp_plotWidget),
    &pappso::MassSpecTracePlotWidget::resolvingPowerComputationSignal,
    this,
    &MassSpecTracePlotCompositeWidget::plotWidgetResolvingPowerComputation);

  connect(mp_plotWidget,
          &pappso::BaseTracePlotWidget::integrationRequestedSignal,
          this,
          &MassSpecTracePlotCompositeWidget::integrationRequested);

  // Create the context menu that should have menu items specific for this mass
  // spec-specific wiget.

  createMainMenu();

  /************ The QCustomPlot widget *************/

  /************ The various widgets in this composite widget ***************/
  /************ The various widgets in this composite widget ***************/

  // The button that triggers the duplication of the trace into another trace
  // plot widget will ask the parent window (MassSpecTracePlotWnd) to perform
  // the duplication.

  connect(m_ui->duplicateTracePushButton, &QPushButton::clicked, [this]() {
    // qDebug();
    static_cast<BaseTracePlotWnd *>(getParentWnd())->duplicateTracePlot(this);
  });
}

void
MassSpecTracePlotCompositeWidget::createMainMenu()
{
  // Complete the base class menu with the color-map-specific menu
  // item.

  qDebug() << "Completing the main menu:" << mp_mainMenu;

  // Debug: the icon is there
  // QIcon the_icon =  m_ui->mainMenuPushButton->icon();
  // QSize size = m_ui->mainMenuPushButton->iconSize();  // typically what you
  // want QPixmap pixmap = the_icon.pixmap(size);
  // pixmap.save("/tmp/massspecicon.png");

  // Update the menu tooltip.
  m_ui->mainMenuPushButton->setToolTip(
    "Mass spec trace plot widget menu (Ctrl+P, M)");

  mp_mainMenu->addSeparator();

  QMenu *deconvolutions_menu_p = mp_mainMenu->addMenu("&Deconvolutions");

  QAction *set_charge_minimal_fractional_part_action_p = new QAction(
    "Set charge &minimal fractional part", dynamic_cast<QObject *>(this));
  set_charge_minimal_fractional_part_action_p->setStatusTip(
    "Set the charge minimal fractional part (typically 0.99)");
  set_charge_minimal_fractional_part_action_p->setShortcut(
    QKeySequence("Ctrl+S, M"));

  connect(
    set_charge_minimal_fractional_part_action_p, &QAction::triggered, [this]() {
      bool ok = false;

      double charge_minimal_fractional_part = QInputDialog::getDouble(
        this,
        "Set the charge minimal fractional part",
        "Charge (z) minimal fractional part (typically 0.99)",
        static_cast<pappso::MassSpecTracePlotWidget *>(mp_plotWidget)
          ->getChargeMinimalFractionalPart(),
        0.5,
        0.9999,
        4,
        &ok);

      if(ok)
        static_cast<pappso::MassSpecTracePlotWidget *>(mp_plotWidget)
          ->setChargeMinimalFractionalPart(charge_minimal_fractional_part);
    });

  deconvolutions_menu_p->addAction(set_charge_minimal_fractional_part_action_p);


  QAction *set_inter_peak_span_action_p = new QAction(
    "Set the span &between two mass peaks", dynamic_cast<QObject *>(this));
  set_inter_peak_span_action_p->setStatusTip(
    "Set the span between two peaks of a charge state envelope (typically 1 or "
    "2)");
  set_inter_peak_span_action_p->setShortcut(QKeySequence("Ctrl+S, B"));

  connect(set_inter_peak_span_action_p, &QAction::triggered, [this]() {
    bool ok = false;

    double peak_span = QInputDialog::getInt(
      this,
      "Set the span between two peaks of a charge state envelope",
      "Span between two peaks (typically 1 or 2)",
      static_cast<pappso::MassSpecTracePlotWidget *>(mp_plotWidget)
        ->getChargeStateEnvelopePeakSpan(),
      1,
      20,
      1,
      &ok);

    if(ok)
      static_cast<pappso::MassSpecTracePlotWidget *>(mp_plotWidget)
        ->setChargeStateEnvelopePeakSpan(peak_span);
  });

  deconvolutions_menu_p->addAction(set_inter_peak_span_action_p);

  mp_mainMenu->addSeparator();

  QAction *gaussian_fit_action_p =
    new QAction("Perform a &Gaussian fit on the visible maxima",
                dynamic_cast<QObject *>(this));
  gaussian_fit_action_p->setShortcut(QKeySequence("Ctrl+G, Ctrl+F"));
  gaussian_fit_action_p->setStatusTip(
    "Perform a Gaussian fit on the visible maxima");

  connect(gaussian_fit_action_p, &QAction::triggered, [this]() {
    gaussianFitOnVisibleMaxima();
  });

  mp_mainMenu->addAction(gaussian_fit_action_p);

  // qDebug() << "EXIT";
}

void
MassSpecTracePlotCompositeWidget::gaussianFitOnVisibleMaxima()
{

  // First we get the source graph that might be of interest to the user.

  QCPAbstractPlottable *source_plottable_p =
    plottableToBeUsedAsIntegrationSource();

  if(source_plottable_p == nullptr)
    {
      QMessageBox::information(this,
                               "Please select *one* trace",
                               "In order to perform a Gaussian fit, the source "
                               "graph needs to be selected.",
                               QMessageBox::StandardButton::Ok,
                               QMessageBox::StandardButton::NoButton);

      return;
    }

  // Thus user has zoomed in such a manner that the monoisotopic pics of a
  // cluster are visible only on their upper part (no points at y=0 should be
  // visible).

  // Get the y range, so that we know what the bottom clipping end is (the floor
  // of the visible data).

  QCPRange plot_widget_x_axis_range = mp_plotWidget->xAxis->range();
  QCPRange plot_widget_y_axis_range = mp_plotWidget->yAxis->range();

  double y_floor_value = plot_widget_y_axis_range.lower;

  pappso::Trace data_trace =
    static_cast<pappso::BaseTracePlotWidget *>(mp_plotWidget)
      ->toTrace(plot_widget_x_axis_range,
                static_cast<QCPGraph *>(source_plottable_p));

  // Now use the Trace filter to get a new trace with the local maxima points
  // only:

  pappso::Trace local_maxima_trace =
    flooredLocalMaxima(data_trace.begin(), data_trace.end(), y_floor_value);

  // We need to iterate in the data in search for all the points that are
  // between x range and in the same time greater or equal to the y floor value.

  // qDebug().noquote() << "\n" << local_maxima_trace.toString() << "\n";

  // At this point we can fit a Gaussian on the trace.

  int i, j, k, n, N;

  std::cout.precision(4); // set precision
  std::cout.setf(std::ios::fixed);

  N = local_maxima_trace.size();

  // qDebug() << "The number of local maxima: " << N;

  double *x     = new double[N];
  double *y     = new double[N];
  double *log_y = new double[N];

  pappso::DataPoint most_intense_datapoint;

  for(size_t iter = 0; iter < local_maxima_trace.size(); ++iter)
    {
      pappso::DataPoint data_point = local_maxima_trace.at(iter);

      if(data_point.y > most_intense_datapoint.y)
        most_intense_datapoint = data_point;

      x[iter] = data_point.x;
      y[iter] = data_point.y;

      // Transform the y values into ln(y) so that we transform something that
      // might be exponential to something that is polynomial (Caruana stuff).

      // std::log is the natural log (ln) in the stdlib, contrary to std::log10.

      log_y[iter] = std::log(data_point.y);
    }

  // Set the polynomial degree to 2 because we want a Gaussian fit and that is
  // what will provide the parameters for the Gaussian fit.
  n = 2;

  double *X = new double[2 * n + 1]; // Array that will store the values of
  // sigma(xi),sigma(xi^2),sigma(xi^3)....sigma(xi^2n)
  for(i = 0; i < 2 * n + 1; i++)
    {
      X[i] = 0;
      for(j = 0; j < N; j++)
        X[i] =
          X[i] + pow(x[j], i); // consecutive positions of the array will store
      // N,sigma(xi),sigma(xi^2),sigma(xi^3)....sigma(xi^2n)
    }

  // B is the Normal matrix(augmented) that will store the
  // equations. So it is an array of arrays.
  double **B = new double *[n + 1];

  for(int iter = 0; iter < n + 1; ++iter)
    B[iter] = new double[n + 2];

  // a is for value of the final coefficient
  double *a = new double[n + 1];

  // Build the Normal matrix by storing the
  // corresponding coefficients at the right positions
  // except the last column of the matrix
  for(i = 0; i <= n; i++)
    for(j = 0; j <= n; j++)
      B[i][j] = X[i + j];

  // Array to store the values of
  // sigma(yi),sigma(xi*yi),sigma(xi^2*yi)...sigma(xi^n*yi)
  double *Y = new double[n + 1];
  for(i = 0; i < n + 1; i++)
    {
      Y[i] = 0;
      for(j = 0; j < N; j++)
        // Y[i]=Y[i]+pow(x[j],i)*y[j];        //consecutive positions will store
        // sigma(yi),sigma(xi*yi),sigma(xi^2*yi)...sigma(xi^n*yi)
        Y[i] =
          Y[i] + pow(x[j], i) * log_y[j]; // consecutive positions will store
      // sigma(yi),sigma(xi*yi),sigma(xi^2*yi)...sigma(xi^n*yi)
    }
  for(i = 0; i <= n; i++)
    B[i][n + 1] = Y[i]; // load the values of Y as the last column of B(Normal
  // Matrix but augmented)
  n = n + 1; // n is made n+1 because the Gaussian Elimination part below was
  // for n equations, but here n is the degree of polynomial and for
  // n degree we get n+1 equations

  std::cout << "\nThe Normal(Augmented Matrix) is as follows:\n";
  for(i = 0; i < n; i++) // print the Normal-augmented matrix
    {
      for(j = 0; j <= n; j++)
        std::cout << B[i][j] << std::setw(16);
      std::cout << "\n";
    }

  for(i = 0; i < n;
      i++) // From now Gaussian Elimination starts(can be ignored) to solve
    // the set of linear equations (Pivotisation)
    for(k = i + 1; k < n; k++)
      if(B[i][i] < B[k][i])
        for(j = 0; j <= n; j++)
          {
            double temp = B[i][j];
            B[i][j]     = B[k][j];
            B[k][j]     = temp;
          }

  for(i = 0; i < n - 1; i++) // loop to perform the gauss elimination
    for(k = i + 1; k < n; k++)
      {
        double t = B[k][i] / B[i][i];
        for(j = 0; j <= n; j++)
          B[k][j] =
            B[k][j] - t * B[i][j]; // make the elements below the pivot elements
                                   // equal to zero or elimnate the variables
      }
  for(i = n - 1; i >= 0; i--) // back-substitution
    { // x is an array whose values correspond to the values of x,y,z..
      a[i] = B[i][n]; // make the variable to be calculated equal to the rhs of
      // the last equation
      for(j = 0; j < n; j++)
        if(j != i) // then subtract all the lhs values except the coefficient of
          // the variable whose value is being calculated
          a[i] = a[i] - B[i][j] * a[j];
      a[i] = a[i] / B[i][i]; // now finally divide the rhs by the coefficient of
                             // the variable to be calculated
    }

  // std::cout << "\nThe values of the coefficients are as follows:\n";
  // for(i = 0; i < n; i++)
  // std::cout << "x^" << i << "=" << a[i]
  //<< std::endl; // Print the values of x^0,x^1,x^2,x^3,....
  // std::cout << "\nHence the fitted Polynomial is given by:\ny=";
  // for(i = 0; i < n; i++)
  // std::cout << " + (" << a[i] << ")"
  //<< "x^" << i;
  // std::cout << "\n";

  double a_coeff        = a[0];
  double b_coeff        = a[1];
  double b_coeff_square = b_coeff * b_coeff;
  double c_coeff        = a[2];

  // std::cout << "a_coeff: " << a_coeff << "\nb_coeff: " << b_coeff
  //<< " b_coeff_square: " << b_coeff_square << "\nc_coeff: " << c_coeff
  //<< std::endl;

  double gaussian_mu = -b_coeff / (2 * c_coeff);
  // double gaussian_mu_square = gaussian_mu * gaussian_mu;

  double gaussian_sigma = sqrt(-1 / (2 * c_coeff));
  double fwhm           = 2 * gaussian_sigma;

  // std::cout << "gaussian_mu: " << gaussian_mu
  //<< "\ngaussian_sigma: " << gaussian_sigma
  //<< " gaussian_sigma_square: " << gaussian_sigma_square << std::endl;

  // Now perform the computation over the x values that were entered
  // initially.

  // From the Hongwei Guo article

  double A = std::exp(a_coeff - (b_coeff_square / (4 * c_coeff)));

  // std::cout
  //<< "Calculating A according to: A = exp(a_coeff - (b_coeff_square / (4 "
  //"* c_coeff)))"
  //<< std::endl;

  // std::cout << "A is now: " << A << std::endl;

  // At this point use our facility to compute the Gaussian trace.

  // This was the old version.
  // libXpertMassCore::MassPeakShaperConfig mass_peak_shaper_config(
  // 0, fwhm, "H", 1, 500, 0, 1,
  // libXpertMassCore::Enums::MassPeakShapeType::GAUSSIAN);

  libXpertMassCore::MassPeakShaperConfig mass_peak_shaper_config;
  mass_peak_shaper_config.setResolution(0);
  mass_peak_shaper_config.setFwhm(fwhm);
  mass_peak_shaper_config.setPointCount(200);
  mass_peak_shaper_config.setReferencePeakMz(most_intense_datapoint.x);
  mass_peak_shaper_config.setMzStep(0);
  mass_peak_shaper_config.setMassPeakShapeType(
    libXpertMassCore::Enums::MassPeakShapeType::GAUSSIAN);

  libXpertMassCore::MassPeakShaper mass_peak_shaper(
    gaussian_mu, A, mass_peak_shaper_config);

  mass_peak_shaper.computePeakShape();

  pappso::Trace fit_trace = mass_peak_shaper.getTrace();

#if 0

  // Testbed for the CBOR classes and the QTcpSocket communications.

  // qDebug() << "Going to write this trace:" << fit_trace.toString();

  libXpertMassCore::MassDataCborMassSpectrumHandler cbor_handler(this, fit_trace);

  QByteArray byte_array;

  cbor_handler.writeByteArray(byte_array);

  qDebug()
    << "Written the Gaussian mass spectrum to a byte array that is now of size:"
    << byte_array.size();

  qDebug() << "Going to signal that the byte array has to be served.";
  getParentWnd()->massDataToBeServedSignal(byte_array);
  qDebug() << "Signalled that the byte array has to be served.";


  // And now read it back from file.
  // cbor_trace_handler.clearTrace();
  // cbor_trace_handler.setInputFileName("/tmp/prova.cbor");
  // cbor_trace_handler.readFile();

  // pappso::Trace read_trace = cbor_trace_handler.getTrace();

  // qDebug() << "Read back this trace:" << read_trace.toString()
  //<< "of mass data type:"
  //<< static_cast<quint64>(cbor_trace_handler.getMassDataType())
  //<< "with title:" << cbor_trace_handler.getTitle();

  // QClipboard *clipboard = QApplication::clipboard();
  // clipboard->setText(read_trace.toString());

#endif

#if 0

    // Original way of doing where we only created points matching the local
    // maxima that were found above.

    for(int iter = 0; iter < N; ++iter)
    {
        double x_minus_mu_square =
            (x[iter] - gaussian_mu) * (x[iter] - gaussian_mu);

        // cout << "x_minus_mu_square: " << x_minus_mu_square << endl;

        double y = A * exp(-x_minus_mu_square / (2 * gaussian_sigma_square));

        std::cout << x[iter] << " " << y << "\n";

        fit_trace.push_back(pappso::DataPoint(x[iter], y));
    }
#endif


#if 0

    pappso::Trace other_fit_trace;

    // All this works well, but we want to use our own code.

    // But we want a much smoother Gaussian curve and larger also, so we manage to
    // compute more points.

    size_t local_maxima_trace_size       = local_maxima_trace.size();
    double local_maxima_x_interval_start = local_maxima_trace.front().x;
    double local_maxima_x_interval_end   = local_maxima_trace.back().x;

    double initial_mz_step =
        (local_maxima_x_interval_end - local_maxima_x_interval_start) /
        local_maxima_trace_size;

    // We want to have a much smaller step:

    double smaller_mz_step = initial_mz_step / 5;

    // We also want more points on the left and on the right of the initial maxima
    // trace x span. For example we want to add a fifth of the initial maxima
    // trace points on the left and the same on the right.

    size_t additional_side_points = local_maxima_trace_size / 2;

    // Now, we want that the Gaussian fit  be symmetrical even if the user did
    // not zoom-in on a symmetric set of isotopic cluster peaks. The symmetry is
    // performed with respect to the center of the Gaussian fit curve:
    // gaussian_mu.

    double left_half_x_interval = gaussian_mu - local_maxima_x_interval_start;
    double right_half_x_interval = local_maxima_x_interval_end - gaussian_mu;

    double half_interval = std::max<double>(left_half_x_interval, right_half_x_interval);

    double wider_x_interval_start =
        gaussian_mu - half_interval - (additional_side_points * smaller_mz_step);

    double wider_x_interval_end =
        gaussian_mu + half_interval + (additional_side_points * smaller_mz_step);

    size_t total_point_count =
        (wider_x_interval_end - wider_x_interval_start) / smaller_mz_step;

    //qDebug().noquote() << "local_maxima_x_interval_start:"
    //<< local_maxima_x_interval_start
    //<< "local_maxima_x_interval_end:"
    //<< local_maxima_x_interval_end << "\n"
    //<< "initial_mz_step:" << initial_mz_step << "\n"
    //<< "smaller_mz_step:" << smaller_mz_step << "\n"
    //<< "additional_side_points:" << additional_side_points
    //<< "\n"
    //<< "wider_x_interval_start:" << wider_x_interval_start
    //<< "\n"
    //<< "wider_x_interval_end:" << wider_x_interval_end << "\n"
    //<< "total_point_count:" << total_point_count << "\n";


    // At this point we can compute all the points of the Gaussian fit curve.

    for(size_t iter = 0; iter < total_point_count; ++iter)
    {
        double mz_value = wider_x_interval_start + (iter * smaller_mz_step);

        //qDebug() << "iter:" << iter << "with mz_value:" << mz_value;

        double mz_value_minus_mu_square =
            (mz_value - gaussian_mu) * (mz_value - gaussian_mu);

        double y =
            A * exp(-mz_value_minus_mu_square / (2 * gaussian_sigma_square));

        //qDebug().noquote() << QString("%1 %2").arg(mz_value, 0, 'f', 6).arg(y)
        //<< "\n";

        other_fit_trace.push_back(pappso::DataPoint(mz_value, y));
    }

#endif

  delete[] x;
  delete[] y;
  delete[] X;
  delete[] Y;

  delete[] a;

  // qDebug() << "n has value:" << n;

  for(int iter = 0; iter < n; ++iter)
    {
      // qDebug() << "Iterating at iter:" << iter;
      delete[] B[iter];
      // qDebug() << "deleted B at iter:" << iter;
    }

  // At this point, add the trace ! But only if it has points in it.

  if(!fit_trace.size())
    {
      // qDebug() << "The Gaussian fit produced an empty trace. Plotting
      // nothing.";

      m_ui->statusLineEdit->setText(
        "The Gaussian fit produced an empty trace. Plotting nothing.");

      // qDebug() << "The Gaussian fit produced an empty trace. Plotting
      // nothing.";

      return;
    }

  // Create a temporary copy of the ProcessingFlow (with no parentship) so that
  // we can modify it with new steps and then pass it as reference to downstream
  // code that will allocate a fully parented instance.

  ProcessingFlow *temp_local_processing_flow_p =
    getCstProcessingFlowPtrForPlottable(source_plottable_p)->clone(nullptr);

  MsRunDataSetCstSPtr ms_run_data_set_csp =
    getMsRunDataSetCstSPtrForPlottable(source_plottable_p);

  addTrace(fit_trace,
           source_plottable_p,
           ms_run_data_set_csp,
           ms_run_data_set_csp->getMsRunId()->getSampleName(),
           *temp_local_processing_flow_p,
           source_plottable_p->pen().color());

  // Now inform the user about the Gaussian data:

  QString msg = QString(
                  "Gaussian fit:\n"
                  "Apex point: (%1,%2)\n"
                  "FWHM: %3\n")
                  .arg(gaussian_mu, 0, 'f', 6)
                  .arg(A, 0, 'f', 2)
                  .arg(fwhm, 0, 'f', 3);

  mp_parentWnd->getProgramWindow()->logColoredTextToConsole(
    msg, source_plottable_p->pen().color());

  delete temp_local_processing_flow_p;

  return;
}

void
MassSpecTracePlotCompositeWidget::plotWidgetKeyPressEvent(
  const pappso::MassSpecTracePlotContext &context)
{
  // qDebug().noquote() << "ENTER with MassSpecTracePlotContext:"
  //                    << context.toString();

  // First call the base class that will fill-in the generic fields if the
  // space char was pressed.
  BaseTracePlotCompositeWidget::plotWidgetKeyPressEvent(context);

  // qDebug() << "After filling-in by base class, the stanza is:"
  //<< m_analysisStanza;

  m_plotWidgetPressedKeyCode = context.m_pressedKeyCode;

  // Now handle here the mass specific data, like z, Mr, m/z.
  if(context.m_pressedKeyCode == Qt::Key_Space)
    {
      QCPAbstractPlottable *plottable_p =
        plottableToBeUsedAsIntegrationSource();

      if(plottable_p == nullptr)
        {
          // qDebug()
          //   << "Returned plottable is nullptr (there might be more than
          //   one).";

#if 0
            // We actually do not do this because that part is handled by the base
            // class.

            // We should inform the user if there are more than one plottable.

            int plottable_count = mp_plotWidget->plottableCount();

            if(plottable_count > 1)
            {
                //qDebug();
                QMessageBox::information(
                    this,
                    "Crafting an analysis stanza for a graph",
                    "Please select a single trace and try again.");
            }
#endif

          // No plottable available, nothing to do.
          return;
        }

      QString stanza = craftAnalysisStanza(context);

      // qDebug() << "analysis stanza:" << stanza;

      // At this point, the analysis stanza can be effectively recorded.

      mp_parentWnd->recordAnalysisStanza(stanza, plottable_p->pen().color());
    }
  // End of
  // if(context.m_pressedKeyCode == Qt::Key_Space)
  else if(context.m_pressedKeyCode == Qt::Key_M)
    {
      // The M (lowercase) keyboard key triggers copying the Mr value to
      // the clipboard when a deconvolution was performed.

      QString mz_value_as_string =
        QString("%1").arg(context.m_lastMr, 0, 'f', 6);

      // qDebug() << "The mz_value_as_string:" << mz_value_as_string;

      QClipboard *clipboard_p = QApplication::clipboard();
      clipboard_p->setText(mz_value_as_string, QClipboard::Clipboard);
    }
  // End of
  // else if(context.m_pressedKeyCode == Qt::Key_M)
  else if(context.m_pressedKeyCode == Qt::Key_Z)
    {
      // The Z (lowercase) keyboard key triggers copying the z value to
      // the clipboard when a deconvolution was performed.

      QString z_value_as_string = QString("%1").arg(context.m_lastZ);

      // qDebug() << "The z_value_as_string:" << z_value_as_string;

      QClipboard *clipboard_p = QApplication::clipboard();
      clipboard_p->setText(z_value_as_string, QClipboard::Clipboard);
    }
  // End of
  // else if(context.m_pressedKeyCode == Qt::Key_Z)
  else if(context.m_pressedKeyCode == Qt::Key_X)
    {
      // The X (lowercase) keyboard key triggers copying the m/z value
      // exactly as the last cursor position (no deconvolution needs to be
      // performed to get to the x value (in a mass spectrum trace, the x
      // value is the m/z value).

      QPointF point = context.m_lastCursorHoveredPoint;

      QString x_value_as_string = QString("%1").arg(point.x(), 0, 'f', 6);

      // qDebug() << "The x_value_as_string:" << x_value_as_string;

      QClipboard *clipboard_p = QApplication::clipboard();
      clipboard_p->setText(x_value_as_string, QClipboard::Clipboard);
    }
  // End of
  // else if(context.m_pressedKeyCode == Qt::Key_X)
}

void
MassSpecTracePlotCompositeWidget::plotWidgetKeyReleaseEvent(
  const pappso::BasePlotContext &context)
{
  BaseTracePlotCompositeWidget::plotWidgetKeyReleaseEvent(context);

  // The release event sets the local key code to 0 so that we know that
  // no keyboard key is being pressed *now*.
  m_plotWidgetPressedKeyCode = context.m_pressedKeyCode;

  // qDebug() << "The key:" << m_plotWidgetPressedKeyCode
  // << "the modifiers:" << context.m_keyboardModifiers;
}

void
MassSpecTracePlotCompositeWidget::plotWidgetMousePressEvent(
  const pappso::MassSpecTracePlotContext &context)
{
  // qDebug() << "The MassSpecTracePlotContext:" << context.toString();

  BaseTracePlotCompositeWidget::plotWidgetMousePressEvent(context);
}

void
MassSpecTracePlotCompositeWidget::plotWidgetMouseReleaseEvent(
  const pappso::MassSpecTracePlotContext &context)
{
  // qDebug() << "The MassSpecTracePlotContext:" << context.toString();

  BaseTracePlotCompositeWidget::plotWidgetMouseReleaseEvent(context);
}

void
MassSpecTracePlotCompositeWidget::plotWidgetMassDeconvolution(
  const pappso::MassSpecTracePlotContext &context)
{
  // qDebug();

  // The point with the append strategy below is that sometimes they get
  // chained ad infinitum if the mouse movements allow calculating
  // spurious deconvolutions... We want to append the deconvolution
  // results to the status text. QString new_text =
  // m_ui->statusLineEdit->text() + QString("-- z: %1 - m/z: %2 - Mr: %3")
  //.arg(context.m_lastZ)
  //.arg(context.m_lastMz)
  //.arg(context.m_lastMr);

  QString new_text =
    QString("[%1--%2] - delta: %3 -- z: %4 - m/z: %5 - Mr: %6")
      .arg(std::min<double>(context.m_xRegionRangeStart,
                            context.m_xRegionRangeEnd),
           0,
           'f',
           libXpertMassCore::MZ_DEC_PLACES)
      .arg(std::max<double>(context.m_xRegionRangeStart,
                            context.m_xRegionRangeEnd),
           0,
           'f',
           libXpertMassCore::MZ_DEC_PLACES)
      .arg(
        fabs(context.m_xDelta), 0, 'f', libXpertMassCore::DELTA_MZ_DEC_PLACES)
      .arg(context.m_lastZ)
      .arg(context.m_lastMz, 0, 'f', libXpertMassCore::MZ_DEC_PLACES)
      .arg(context.m_lastMr, 0, 'f', libXpertMassCore::MZ_DEC_PLACES);

  m_ui->statusLineEdit->setText(new_text);
}

void
MassSpecTracePlotCompositeWidget::plotWidgetResolvingPowerComputation(
  const pappso::MassSpecTracePlotContext &context)
{
  // We want to append the resolving power results to the status text.
  QString new_text = m_ui->statusLineEdit->text() +
                     QString("-- Rp: %1").arg(context.m_lastResolvingPower);

  m_ui->statusLineEdit->setText(new_text);
}

void
MassSpecTracePlotCompositeWidget::lastCursorHoveredPoint(const QPointF &pointf)

{
  BaseTracePlotCompositeWidget::lastCursorHoveredPoint(pointf);

  // qDebug() << "pointf:" << pointf;

  // At this point we can start doing mass spectrum-specific calculations.
}

void
MassSpecTracePlotCompositeWidget::moveMouseCursorToNextGraphPoint(
  const pappso::BasePlotContext &context)
{
  BaseTracePlotCompositeWidget::moveMouseCursorToNextGraphPoint(context);

  // Modify the context to conform to the integration range that was
  // computed in the base class function.

  pappso::BasePlotContext local_context;
  local_context.initialize(context);
  local_context.m_xRegionRangeStart = m_integrationRange.lower;
  local_context.m_xRegionRangeEnd   = m_integrationRange.upper;

  if(m_isSinglePointIntegrationMode)
    integrationRequested(local_context);
}

void
MassSpecTracePlotCompositeWidget::integrationRequested(
  const pappso::BasePlotContext &context)
{
  // This function is called when an integration is performed
  // *starting* from this widget, not ending in this widget.
  // For integrations that are directed to a mass spectrum,
  // check the MassSpectracePlotWnd class !!!

  // qDebug().noquote() << "context:" << context.toString();

  // The reference implementation is the
  // TicXicChromTracePlotCompositeWidget one. Look there for all the
  // explanatory comments.

  // Make a local copy of the context because we may need to modify it.
  pappso::BasePlotContext local_context;
  local_context.initialize(context);

  // qDebug().noquote() << "local context:" << local_context.toString();

  // Check if the integration scope contains meaningful data. We are in trace
  // plot widget, so the scope might be in the form of a
  // mono-dimensional scope, although this is not mandatory.

  if(local_context.msp_integrationScope->is2D())
    {
      // qDebug()
      //   << "Odd situation where the selection polygon is two-dimensional "
      //      "and the trace plot selection is mono-dimensional. Converting "
      //      "it to a 1D selection polygon.";

      //**local_context.msp_integrationScope->convertTo1D();
      double width;
      if(!local_context.msp_integrationScope->getWidth(width))
        qFatal() << "Failed to get width.";

      QPointF point;
      if(!local_context.msp_integrationScope->getPoint(point))
        qFatal() << "Failed to get point.";

      local_context.msp_integrationScope =
        std::make_shared<const pappso::IntegrationScopeBase>(
          pappso::IntegrationScope(point, width));
    }

  std::const_pointer_cast<pappso::IntegrationScopeBase>(
    local_context.msp_integrationScope)
    ->setDataKindX(pappso::Enums::DataKind::mz);

  // Sanity check.

  // It might happen that the user inadventently starts a selection
  // left of x = 0 (that is, at negative m/z values), so we cope with
  // this, but it makes absolutely no sense that the whole selected
  // area be in the negative region of the m/z scale.
  double range_start;
  double range_end;

  local_context.msp_integrationScope->range(
    pappso::Enums::Axis::x, range_start, range_end);

  // qDebug().noquote() << "local context extracted X axis range:"
  //                    << range_start << "-" << range_end;

  // Having negative masses does not make any sense.
  if(range_start < 0 && range_end < 0)
    return;

  QCPAbstractPlottable *plottable_p = plottableToBeUsedAsIntegrationSource();

  if(plottable_p == nullptr)
    {
      QMessageBox::information(this,
                               "Please select *one* trace",
                               "In order to perform an integration, the source "
                               "graph needs to be selected.",
                               QMessageBox::StandardButton::Ok,
                               QMessageBox::StandardButton::NoButton);

      return;
    }

  // Create a temporary copy of the ProcessingFlow (with no parentship) so that
  // we can modify it with new steps and then pass it as reference to downstream
  // code that will allocate a fully parented instance.

  ProcessingFlow *temp_processing_flow_p =
    getProcessingFlowPtrForPlottable(plottable_p)->clone(nullptr);

  // qDebug().noquote() << "Got processing flow from plottable:\n"
  //                    << temp_processing_flow_p->toString(0, "  ");

  MsRunDataSetCstSPtr ms_run_data_set_csp =
    getMsRunDataSetCstSPtrForPlottable(plottable_p);

  // qDebug().noquote() << "The ms run data statistics:"
  //                    <<
  //                    ms_run_data_set_csp->getMsRunDataSetStats().toString();

  // Create a processing step to document the specifics of this new
  // integration.

  ProcessingStepSPtr processing_step_sp = std::make_shared<ProcessingStep>();

  MsFragmentationSpec ms_fragmentation_spec =
    temp_processing_flow_p->getDefaultMsFragmentationSpec();

  // Only set the ms frag spec to the processing spec if it is valid.
  if(ms_fragmentation_spec.isValid())
    {
      // qDebug().noquote() << "The default ms frag spec from the
      // processing flow " "is valid, using it : "
      //<< ms_fragmentation_spec.toString();

      processing_step_sp->setMsFragmentationSpec(ms_fragmentation_spec);
    }
  else
    {
      // qDebug() << "The default ms frag spec from the processing flow is
      // not
      // "
      //"valid. Not using it.";
    }

  // Now make sure we document the integration scope.
  // Now define the graph range for the integration operation.
  processing_step_sp->setIntegrationScope(local_context.msp_integrationScope);

  // qDebug() << "Set this integration scope to the processing step:"
  //          << processing_step_sp->getIntegrationScope()->toString();

  // We must say what is the data source processing type.
  processing_step_sp->setSrcProcessingType(pappso::Enums::Axis::x, "MZ");

  // It is essential that we document the relation between the axes and
  // the corresponding data type.
  processing_step_sp->setDataKind(pappso::Enums::Axis::x,
                                  pappso::Enums::DataKind::mz);
  processing_step_sp->setDataKind(pappso::Enums::Axis::y,
                                  pappso::Enums::DataKind::unset);

  if(m_ui->integrateToRtPushButton->isChecked())
    {
      // qDebug() << "Integrating to XIC chromatogram.";

      processing_step_sp->setDestProcessingType("RT");

      // Should perform automatically the conversion from non-const to const.
      temp_processing_flow_p->getStepsRef().push_back(processing_step_sp);

      static_cast<MassSpecTracePlotWnd *>(mp_parentWnd)
        ->integrateToRt(plottable_p, *temp_processing_flow_p);
    }
  else if(m_ui->integrateToMzPushButton->isChecked() ||
          m_ui->integrateToDtMzPushButton->isChecked() ||
          m_ui->integrateToMzRtPushButton->isChecked())
    {
      if(m_ui->integrateToMzPushButton->isChecked())
        {
          // qDebug() << "Integrating to mass spectrum.";
          processing_step_sp->setDestProcessingType("MZ");
        }
      else if(m_ui->integrateToMzRtPushButton->isChecked())
        {
          // qDebug() << "Integrating to rt|m/z color map.";
          processing_step_sp->setDestProcessingType("MZ_RT");
        }
      else if(m_ui->integrateToDtMzPushButton->isChecked())
        {
          // qDebug() << "Integrating to dt|m/z color map.";
          processing_step_sp->setDestProcessingType("DT_MZ");
        }

      // At this point we need to define how the mass data integrator will
      // combine spectra, since we are integrating to a mass spectrum.

      // The function below will first get the potential settings that might
      // have been stored by the user when "Set as default" is checked in the
      // MzIntegrationParamsDlg window. If no integration settings were found,
      // then other possibilities are tested (in particular by looking into
      // the ProcessingFlow).

      // Remember, however, that the MzIntegrationParams that are returned
      // might have been stored as settings for a totally different
      // data set. So we need to adjust the m/z data in them:

      // temp_processing_flow_p was obtained above
      // for the plottable we are working on.
      // Heap-allocated !!
      pappso::MzIntegrationParams *mz_integration_params_p =
        temp_processing_flow_p->getSensibleMzIntegrationParams();

      // qDebug().noquote() << "Got sensible mz integration params:\n"
      //                    << mz_integration_params_p->toString();

      // Note that we cannot integrate mass spectra with a high resolution
      // here. We need to reduce the resolution to integer resolution.
      if(m_ui->integrateToDtMzPushButton->isChecked() ||
         m_ui->integrateToMzRtPushButton->isChecked())
        {
          mz_integration_params_p->setBinningType(
            pappso::MzIntegrationParams::BinningType::NONE);
          mz_integration_params_p->setDecimalPlaces(0);
        }

      // We finally have mz integ params that we can set to the processing
      // step that we are creating to document this current integration.
      // Takes ownership
      processing_step_sp->setMzIntegrationParams(mz_integration_params_p);

      // And nwo add the new processing step to the local copy of the
      // processing flow, so that we document on top of all the previous
      // steps, also this last one.

      // Should perform automatically the conversion from non-const to const.
      temp_processing_flow_p->getStepsRef().push_back(processing_step_sp);

      // qDebug().noquote() << "Pushed back new processing step:\n"
      //                    << processing_step_sp->toString();

      // The local processing flow contains all the previous steps and the
      // last one that documents this current integration.

      if(m_ui->integrateToMzPushButton->isChecked())
        {
          static_cast<MassSpecTracePlotWnd *>(mp_parentWnd)
            ->integrateToMz(plottable_p, nullptr, *temp_processing_flow_p);
        }
      else if(m_ui->integrateToDtMzPushButton->isChecked())
        {
          static_cast<MassSpecTracePlotWnd *>(mp_parentWnd)
            ->integrateToDtMz(plottable_p, *temp_processing_flow_p);
        }
      else if(m_ui->integrateToMzRtPushButton->isChecked())
        {
          static_cast<MassSpecTracePlotWnd *>(mp_parentWnd)
            ->integrateToMzRt(plottable_p, *temp_processing_flow_p);
        }
    }
  else if(m_ui->integrateToDtPushButton->isChecked())
    {
      // qDebug() << "Integrating to drift spectrum.";

      processing_step_sp->setDestProcessingType("DT");

      // Should perform automatically the conversion from non-const to const.
      temp_processing_flow_p->getStepsRef().push_back(processing_step_sp);

      static_cast<MassSpecTracePlotWnd *>(mp_parentWnd)
        ->integrateToDt(plottable_p, *temp_processing_flow_p);
    }
  else if(m_ui->integrateToDtRtPushButton->isChecked())
    {
      // qDebug() << "Integrating to TIC|XIC chromatogram / drift spectrum
      // colormap.";

      processing_step_sp->setDestProcessingType("DT_RT");

      // Should perform automatically the conversion from non-const to const.
      temp_processing_flow_p->getStepsRef().push_back(processing_step_sp);

      static_cast<MassSpecTracePlotWnd *>(mp_parentWnd)
        ->integrateToDtRt(plottable_p, *temp_processing_flow_p);
    }
  else if(m_ui->integrateToIntPushButton->isChecked())
    {
      // qDebug() << "Integrating to TIC intensity.";

      processing_step_sp->setDestProcessingType("INT");

      // Should perform automatically the conversion from non-const to const.
      temp_processing_flow_p->getStepsRef().push_back(processing_step_sp);

      static_cast<BasePlotWnd *>(mp_parentWnd)
        ->integrateToTicIntensity(
          plottable_p, nullptr, *temp_processing_flow_p);
    }
  else
    {
      QMessageBox::information(
        this,
        "Please select a destination for the integration",
        "In order to perform an integration, select a destination "
        "like a mass spectrum or a drift spectrum by clicking one "
        "of the buttons on the right margin of the plot widget.",
        QMessageBox::StandardButton::Ok,
        QMessageBox::StandardButton::NoButton);
    }

  delete temp_processing_flow_p;

  return;
}

QCPGraph *
MassSpecTracePlotCompositeWidget::addTrace(
  const pappso::Trace &trace,
  QCPAbstractPlottable *parent_plottable_p,
  MsRunDataSetCstSPtr ms_run_data_set_csp,
  const QString &sample_name,
  const ProcessingFlow &processing_flow,
  const QColor &color)
{

  qDebug();

  // If color is not valid, we need to create one.

  QColor local_color(color);

  // Get color from the available colors, or if none is available, create
  // one randomly without requesting the user to select one from
  // QColorDialog.
  if(!local_color.isValid())
    local_color = libXpertMassGui::ColorSelector::getColor(true);

  // qDebug().noquote() << "The processing flow:" <<
  // processing_flow.toString();

  // The QCPGraph that is going to be created will have the proper color
  // defined in it. Also, we grab the graph that has been created because
  // we need to associate it with the matching processing flow in the map.

  // We have to verify if there is a destination policy for this composite
  // plot widget. There are different situations.

  // 1. The erase trace and create new push button is checked: the user
  // want to remove all the selected traces and replace them with the
  // trace passed as parameter.

  // 2. The keep traces and combine push button is checked: the user want
  // to combine all the selected traces individually to the new trace
  // passed as parameters.

  // 3. If no destination policy is defined, simply add the new trace
  // without touching any other trace that might be already plotted there.

  // We'll need this to factorize code.
  bool must_create_new_graph    = false;
  QCPAbstractPlottable *graph_p = nullptr;
  QList<QCPAbstractPlottable *> dest_plottable_list =
    plottablesToBeUsedAsIntegrationDestination();

  // qDebug() << "The destination plottable size has "
  //<< dest_plottable_list.size() << "graphs.";

  if(m_newTraceHandlingMethod == NewTraceHandlingMethod::OVERLAY)
    {
      // Keep the traces there and simply add a new one.
      must_create_new_graph = true;
    }
  else if(m_newTraceHandlingMethod == NewTraceHandlingMethod::MINUS_COMBINE ||
          m_newTraceHandlingMethod == NewTraceHandlingMethod::PLUS_COMBINE)

    {
      // qDebug();

      // Combination is requested. There are some situations:

      // 1. If there is no single trace in the widget, then simply add a
      // new trace.
      //
      // 2. If there is a single trace, then combine with that, selected
      // or not.
      //
      // 3. If there are multiple traces, then only combine with those.


      // Since we are asked that a combination to either a trace or a mass
      // spectrum be performed, there must be a non-nullptr mz integration
      // pointer in the step.
      // Returns the pointer, not a newly allocated instance for us.

      // If the user has configured that no binning is to be performed, we
      // should get that config now (or any other, of course).
      const pappso::MzIntegrationParams *mz_integration_params_p =
        processing_flow.getSensibleMzIntegrationParams();

      // qDebug() << "The sensible MzIntegrationParams: "
      //          << mz_integration_params_p->toString();

      if(mz_integration_params_p == nullptr)
        qFatal(
          "Combining to a mass spectrum, cannot be that the mz "
          "integration params are not available in the processing flow.");

      if(!dest_plottable_list.size())
        {
          // qDebug() << "Adding trace but there is not a single trace
          // available, " "creating a new one.";

          must_create_new_graph = true;
        }
      else
        {
          // qDebug() << "Adding new trace with combination requested.";

          // qDebug() << "The most recent mz integration parameters:"
          //          << mz_integration_params_p->toString(0, " ");

          if(mz_integration_params_p->getBinningType() ==
             pappso::MzIntegrationParams::BinningType::NONE)
            {
              // qDebug() << "Binning type none.";

              pappso::TraceMinusCombiner trace_minus_combiner;
              pappso::TracePlusCombiner trace_plus_combiner;

              // No binning is required, so most simple situation.

              for(int iter = 0; iter < dest_plottable_list.size(); ++iter)
                {
                  QCPAbstractPlottable *iter_plottable_p =
                    dest_plottable_list.at(iter);
                  pappso::Trace iter_trace =
                    static_cast<pappso::BaseTracePlotWidget *>(mp_plotWidget)
                      ->toTrace(dynamic_cast<QCPGraph *>(iter_plottable_p));

                  // Initialize the receiving map trace with the trace we
                  // are iterating into.
                  pappso::MapTrace map_trace(iter_trace);

                  // And now combine the trace we got as parameter.

                  if(m_newTraceHandlingMethod ==
                     NewTraceHandlingMethod::MINUS_COMBINE)
                    {
                      // qDebug() << "Now minus combining without "
                      //             "binning.";
                      trace_minus_combiner.combine(map_trace, trace);
                    }
                  else if(m_newTraceHandlingMethod ==
                          NewTraceHandlingMethod::PLUS_COMBINE)
                    {
                      // qDebug() << "Now minus combining without "
                      //             "binning.";
                      trace_plus_combiner.combine(map_trace, trace);
                    }

                  // Old version that was deprecated in 5.14.2
                  //
                  // static_cast<QCPGraph *>(iter_plottable_p)
                  //->setData(
                  // QVector<double>::fromStdVector(map_trace.firstToVector()),
                  // QVector<double>::fromStdVector(
                  // map_trace.secondToVector()),
                  // true);

                  QVector<double> key_vector;
                  QVector<double> value_vector;

                  // Apparently, as of 20210928, that bug is fixed.
                  // #pragma GCC warning "Filippo Rusconi: Please check if
                  // the
                  // bug was fixed in Qt"

                  // Now replace the graph's data. Note that the data are
                  // inherently sorted (true below).

                  // The begin() -- end() ranges constructor did not work
                  // as of Qt 5.14.2 this day: 20200721 -- and still does
                  // not work 20211018.

                  // key_vector   = QVector(map_trace.xValues().begin(),
                  // map_trace.xValues().end());
                  // value_vector = QVector(map_trace.yValues().begin(),
                  // map_trace.yValues().end());


                  for(auto &value : map_trace.xValues())
                    key_vector.push_back(value);

                  for(auto &value : map_trace.yValues())
                    value_vector.push_back(value);

                  static_cast<QCPGraph *>(iter_plottable_p)
                    ->setData(key_vector, value_vector, true);
                }

              // Note how we do not change neither the sample name text nor its
              // color. He we just increment/decrement a trace. We do not set a
              // new trace.
            }
          else if(mz_integration_params_p->getBinningType() ==
                  pappso::MzIntegrationParams::BinningType::ARBITRARY)
            {
              // qDebug() << "Binning type arbitrary.";
              pappso::MassSpectrumMinusCombiner mass_spectrum_minus_combiner;
              pappso::MassSpectrumPlusCombiner mass_spectrum_plus_combiner;

              // There is a requirement to perform a binning. We thus need
              // to create the bins. The smallest mz step needs to be the
              // smallest of the trace we get as a parameter and of the
              // trace we got for the receiving graph.

              // We'll need the mz integration parameters from the
              // processing flow we  get as parameters. Make a copy of
              // them as we'll need to set the mz smallest and greatest
              // values to them.

              pappso::MzIntegrationParams *copy_mz_integration_params_p =
                mz_integration_params_p->clone(nullptr);

              // qDebug() << "Combinining new trace into a graph trace.";

              for(int iter = 0; iter < dest_plottable_list.size(); ++iter)
                {
                  QCPAbstractPlottable *iter_plottable_p =
                    dest_plottable_list.at(iter);
                  pappso::Trace iter_dest_trace =
                    static_cast<pappso::BaseTracePlotWidget *>(mp_plotWidget)
                      ->toTrace(static_cast<QCPGraph *>(iter_plottable_p));

                  // Initialize the receiving map trace with the trace we
                  // are iterating into.
                  pappso::MapTrace map_trace(iter_dest_trace);

                  // In order to maintain the phasing of bins, we should get the
                  // first m/z value of the destination trace and use it as an
                  // anchor point to the creation of the bins.

                  double src_smallest_mz  = trace.front().x;
                  double dest_smallest_mz = iter_dest_trace.front().x;

                  copy_mz_integration_params_p->setSmallestMz(src_smallest_mz);
                  copy_mz_integration_params_p->setLowerAnchorMz(
                    dest_smallest_mz);

                  double src_greatest_mz  = trace.back().x;
                  double dest_greatest_mz = iter_dest_trace.back().x;

                  copy_mz_integration_params_p->setGreatestMz(src_greatest_mz);
                  copy_mz_integration_params_p->setUpperAnchorMz(
                    dest_greatest_mz);

                  // qDebug()
                  //   << "At this point, the mz integration params:"
                  //   << copy_mz_integration_params_p->toString(0, " ");

                  // At this point we can ask the bins to be computed.
                  std::vector<double> bins =
                    copy_mz_integration_params_p->createBins();

                  // qDebug() << "Prepared bins with " << bins.size() <<
                  // "elements."
                  // << "starting with mz" << bins.front() << "ending with mz"
                  // << bins.back();

                  // Now set the bins to the combiner.

                  if(m_newTraceHandlingMethod ==
                     NewTraceHandlingMethod::MINUS_COMBINE)
                    {
                      // qDebug() <<
                      // "NewTraceHandlingMethod::MINUS_COMBINE";
                      mass_spectrum_minus_combiner.setBins(bins);
                      mass_spectrum_minus_combiner.combine(map_trace, trace);
                    }
                  else if(m_newTraceHandlingMethod ==
                          NewTraceHandlingMethod::PLUS_COMBINE)
                    {
                      // qDebug() <<
                      // "NewTraceHandlingMethod::PLUS_COMBINE";
                      mass_spectrum_plus_combiner.setBins(bins);
                      mass_spectrum_plus_combiner.combine(map_trace, trace);
                    }


                  // Since the combination is terminated, set the new data
                  // to the graph (the data are sorted, thus the true
                  // below).

                  // Old version that was deprecated in 5.14.2
                  //
                  // static_cast<QCPGraph *>(iter_plottable_p)
                  //->setData(
                  // QVector<double>::fromStdVector(map_trace.firstToVector()),
                  // QVector<double>::fromStdVector(
                  // map_trace.secondToVector()),
                  // true);

                  QVector<double> key_vector;
                  QVector<double> value_vector;

                  // Apparently, as of 20210928, that bug is fixed.
                  // #pragma GCC warning "Filippo Rusconi: Please check if
                  // the
                  // bug was fixed in Qt"

                  // Now replace the graph's data. Note that the data are
                  // inherently sorted (true below).

                  // The begin() -- end() ranges constructor did not work
                  // as of Qt 5.14.2 this day: 20200721 -- and still does
                  // not work 20211018.


                  // key_vector   = QVector(map_trace.xValues().begin(),
                  // map_trace.xValues().end());
                  // value_vector = QVector(map_trace.yValues().begin(),
                  // map_trace.yValues().end());

                  // This is the workaround fix.

                  for(auto &value : map_trace.xValues())
                    key_vector.push_back(value);

                  for(auto &value : map_trace.yValues())
                    value_vector.push_back(value);

                  static_cast<QCPGraph *>(iter_plottable_p)
                    ->setData(key_vector, value_vector, true);
                }
              // End of
              // for(int iter = 0; iter < dest_plottable_list.size(); ++iter)

              // Note how we do not change neither the sample name text nor its
              // color. He we just increment/decrement a trace. We do not set a
              // new trace.
            }
          else if(mz_integration_params_p->getBinningType() ==
                  pappso::MzIntegrationParams::BinningType::DATA_BASED)
            {
              // The existing trace should serve as a data template to construct
              // bins for our new trace to be combined to the existing one.

              qDebug() << "Binning type data-based.";

              pappso::MassSpectrumMinusCombiner mass_spectrum_minus_combiner;
              pappso::MassSpectrumPlusCombiner mass_spectrum_plus_combiner;

              // There is a requirement to perform a binning. We thus need
              // to create the bins. The smallest mz step needs to be the
              // smallest of the trace we get as a parameter and of the
              // trace we got for the receiving graph.

              // We'll need the mz integration parameters from the
              // processing flow we  get as parameters. Make a copy of
              // them as we'll need to set the mz smallest and greatest
              // values to them.

              pappso::MzIntegrationParams *copy_mz_integration_params_p =
                mz_integration_params_p->clone(nullptr);

              // qDebug() << "Combinining new trace into a graph trace.";

              for(int iter = 0; iter < dest_plottable_list.size(); ++iter)
                {
                  QCPAbstractPlottable *iter_plottable_p =
                    dest_plottable_list.at(iter);
                  pappso::Trace iter_dest_trace =
                    static_cast<pappso::BaseTracePlotWidget *>(mp_plotWidget)
                      ->toTrace(static_cast<QCPGraph *>(iter_plottable_p));

                  // Initialize the receiving map trace with the trace we
                  // are iterating into.
                  pappso::MapTrace map_trace(iter_dest_trace);

                  // In order to maintain the phasing of bins, we should get the
                  // first m/z value of the destination trace and use it as an
                  // anchor point to the creation of the bins.

                  double src_smallest_mz = trace.front().x;
                  copy_mz_integration_params_p->setSmallestMz(src_smallest_mz);

                  double src_greatest_mz = trace.back().x;
                  copy_mz_integration_params_p->setGreatestMz(src_greatest_mz);

                  qDebug() << qSetRealNumberPrecision(6)
                           << "The source plot trace has left m/z:"
                           << src_smallest_mz
                           << "and right m/z:" << src_greatest_mz;

                  pappso::MassSpectrumCstSPtr dest_mass_spectrum_csp =
                    std::make_shared<const pappso::MassSpectrum>(
                      iter_dest_trace);

                  qDebug() << qSetRealNumberPrecision(6)
                           << "The destination plot trace has left m/z:"
                           << dest_mass_spectrum_csp->minX() << "and right m/z:"
                           << dest_mass_spectrum_csp->maxX();

                  // qDebug()
                  //   << "At this point, the mz integration params:"
                  //   << copy_mz_integration_params_p->toString(0, " ");

                  // At this point we can ask the bins to be computed.
                  // These bins should be phased with the bins in
                  // iter_dest_trace.
                  std::vector<double> bins =
                    copy_mz_integration_params_p->createBins(
                      dest_mass_spectrum_csp);

                  // qDebug() << "Prepared bins with " << bins.size() <<
                  // "elements."
                  // << "starting with mz" << bins.front() << "ending with mz"
                  // << bins.back();

                  // Now set the bins to the combiner.

                  if(m_newTraceHandlingMethod ==
                     NewTraceHandlingMethod::MINUS_COMBINE)
                    {
                      // qDebug() <<
                      // "NewTraceHandlingMethod::MINUS_COMBINE";
                      mass_spectrum_minus_combiner.setBins(bins);
                      mass_spectrum_minus_combiner.combine(map_trace, trace);
                    }
                  else if(m_newTraceHandlingMethod ==
                          NewTraceHandlingMethod::PLUS_COMBINE)
                    {
                      // qDebug() <<
                      // "NewTraceHandlingMethod::PLUS_COMBINE";
                      mass_spectrum_plus_combiner.setBins(bins);
                      mass_spectrum_plus_combiner.combine(map_trace, trace);
                    }


                  // Since the combination is terminated, set the new data
                  // to the graph (the data are sorted, thus the true
                  // below).

                  // Old version that was deprecated in 5.14.2
                  //
                  // static_cast<QCPGraph *>(iter_plottable_p)
                  //->setData(
                  // QVector<double>::fromStdVector(map_trace.firstToVector()),
                  // QVector<double>::fromStdVector(
                  // map_trace.secondToVector()),
                  // true);

                  QVector<double> key_vector;
                  QVector<double> value_vector;

                  // Apparently, as of 20210928, that bug is fixed.
                  // #pragma GCC warning "Filippo Rusconi: Please check if
                  // the
                  // bug was fixed in Qt"

                  // Now replace the graph's data. Note that the data are
                  // inherently sorted (true below).

                  // The begin() -- end() ranges constructor did not work
                  // as of Qt 5.14.2 this day: 20200721 -- and still does
                  // not work 20211018.


                  // key_vector   = QVector(map_trace.xValues().begin(),
                  // map_trace.xValues().end());
                  // value_vector = QVector(map_trace.yValues().begin(),
                  // map_trace.yValues().end());

                  // This is the workaround fix.

                  for(auto &value : map_trace.xValues())
                    key_vector.push_back(value);

                  for(auto &value : map_trace.yValues())
                    value_vector.push_back(value);

                  static_cast<QCPGraph *>(iter_plottable_p)
                    ->setData(key_vector, value_vector, true);
                }
              // End of
              // for(int iter = 0; iter < dest_plottable_list.size(); ++iter)
            }
          else
            {
              qFatal()
                << "Programming error. This line should never be reached.";
            }
        }
      // End of
      // else of if(!graph_count), that if there are graphs in the plot
      // widget.
    }
  // End of if(m_ui->keepTraceCombineNewPushButton->isChecked())
  else if(m_newTraceHandlingMethod == NewTraceHandlingMethod::REPLACE)
    {
      // qDebug() << "Adding new trace by the replace method: removing the
      // "
      //"current one(s).";

      // Start by erasing all the traces that are selected.

      for(int iter = 0; iter < dest_plottable_list.size(); ++iter)
        {
          QCPAbstractPlottable *iter_plottable_p = dest_plottable_list.at(iter);

          // Destroy the graph , but not recursively (false).
          mp_parentWnd->getProgramWindow()->plottableDestructionRequested(
            this, iter_plottable_p, false);
        }

      must_create_new_graph = true;
    }
  else
    {
      // qDebug();

      // None of the button were checked, we simply created a new graph.
      must_create_new_graph = true;
    }

  if(must_create_new_graph)
    {
      // At this point we know we need to create a new trace.

      // qDebug()
      //<< "Adding a new trace: in fact we need to create one anew as
      // is.";

      graph_p = static_cast<pappso::BaseTracePlotWidget *>(mp_plotWidget)
                  ->addTrace(trace, local_color);

      // Each time a new trace is added anywhere, it needs to be
      // documented in the main program window's graph nodes tree! This is
      // what we call the ms run data plot graph filiation documentation.

      mp_parentWnd->getProgramWindow()->documentMsRunDataPlottableFiliation(
        ms_run_data_set_csp, graph_p, parent_plottable_p, this);

#if 0
        //////// CAUTION ///////
        //////// CAUTION ///////
        // Erroneous: if we make the connection below each time a new graph is added
        // to a plot widget, the sigma function will be called as many times for each
        // graph that is being destroyed ! Now there is a single connection for the
        // pappso::BaseTracePlotWidget in the addTracePlot() function in each
        // XxxPlotWnd class.

        // The single graph destruction signal.

        //qDebug().noquote()
        //<< "Making the connection:"
        //<< "pappso::BaseTracePlotWidget::graphDestructionRequestedSignal to "
        //"mp_parentWnd->getProgramWindow()->graphDestructionRequested.";

        //connect(getPlotWidget(),
        //&pappso::BaseTracePlotWidget::graphDestructionRequestedSignal,
        //[this](
        //const pappso::BaseTracePlotWidget *base_trace_plot_widget_p,
        //QCPGraph *graph_p,
        //const pappso::BasePlotContext &context) {
        //mp_parentWnd->getProgramWindow()->graphDestructionRequested(this, graph_p, context);
        //});
        //////// CAUTION ///////
        //////// CAUTION ///////
#endif

      // We need to allocate a new ProcessingFlow instance with the graph
      // as its parent.
      ProcessingFlow *processing_flow_p = processing_flow.clone(graph_p);

      // Map the graph_p to the processing flow that we got to document
      // how the trace was obtained in the first place.
      m_plottableProcessingFlowMap[graph_p] = processing_flow_p;

      // Note how we do  change the sample name text and its color.
      // He we do set a new trace.
      m_ui->sampleNameLabel->setText(sample_name);
      setSampleNameColor(color);
    }
  // End of
  // if(must_create_new_graph)

  // We need to rescale the axes so that the trace is entirely shown!
  // mp_plotWidget->rescaleAxes();

  mp_plotWidget->replot();

  // graph_p might be nullptr.
  return static_cast<QCPGraph *>(graph_p);
}

QString
MassSpecTracePlotCompositeWidget::craftAnalysisStanza(
  const pappso::MassSpecTracePlotContext &context)
{
  // qDebug().noquote() << "With context:" << context.toString();
  // qDebug().noquote()
  //   << "We get from the base class stanza work the following text:"
  //   << m_analysisStanza;

  QCPAbstractPlottable *plottable_p = plottableToBeUsedAsIntegrationSource();

  if(plottable_p == nullptr)
    {
      // qDebug()
      //   << "Returned plottable is nullptr (there might be more than one).";

      // We should inform the user if there are more than one plottable.

      int plottable_count = mp_plotWidget->plottableCount();

      if(plottable_count > 1)
        {
          // qDebug();
          QMessageBox::information(
            this,
            "Crafting an analysis stanza for a graph",
            "Please select a single trace and try again.");
        }

      m_analysisStanza = "";
      return QString();
    }

  // qDebug() << "Plottable:" << plottable_p;

  MsRunDataSetCstSPtr ms_run_data_set_csp =
    getCstProcessingFlowPtrForPlottable(plottable_p)->getMsRunDataSetCstSPtr();

  // We do only handle here the specialty format elements: %z and %M.
  // Note that %M is a double value that may accept a precision format string
  // like this:
  // %M.3 to indicate that the molecular mass should be printed out with three
  // decimals.

  // The m_analysisStanza member has been filled-in by the base class and
  // the specialty format strings %z and %M have been let untouched, so we can
  // now work on them.

  // So, in fact, the m_analysisStanza behaves like a format string itself.

  QString formatString = m_analysisStanza;

  static const QRegularExpression re("%([zM])(\\.(\\d+))?");

  // Not all the format specifications are for number, specify which are
  static const QSet<QChar> floatKeys   = {'M'};
  static const QSet<QChar> integerKeys = {'z'};
  static const QSet<QChar> stringKeys  = {};

  QHash<QChar, QVariant> replacements;

  /////////////////// THE z charge value //////////////////////

  if(!qIsNaN(context.m_lastMz))
    replacements['M'] = context.m_lastMr;

  if(context.m_lastZ != std::numeric_limits<quint16>::max())
    replacements['z'] = context.m_lastZ;

  QString stanza;
  stanza.reserve(formatString.size());

  int lastIndex = 0;
  auto it       = re.globalMatch(formatString);

  while(it.hasNext())
    {
      QRegularExpressionMatch m = it.next();

      int start = m.capturedStart();
      int end   = m.capturedEnd();

      // Copy text up to this match
      stanza += formatString.mid(lastIndex, start - lastIndex);

      const QChar key = m.captured(1)[0];

      qDebug() << "Captured key: " << key;

      const QString precision_spec_string = m.captured(3); // just the digits
      bool ok                             = false;
      int precision_spec                  = precision_spec_string.toInt(&ok);

      if(!ok)
        precision_spec = 0;

      if(!replacements.contains(key))
        {
          QString untouched = m.captured(0);

          qDebug() << "Adding new untouched string element:" << untouched;

          stanza    += untouched;
          lastIndex  = end;
          continue;
        }

      const QVariant &v = replacements[key];

      if(floatKeys.contains(key))
        {
          // ---- FLOAT TOKENS ----
          double d = 0.0;

          if(v.canConvert<double>())
            d = v.toDouble(); // integer or float → OK
          else
            {
              stanza    += "<NaN>"; // or keep literal, your choice
              lastIndex  = end;
              continue;
            }

          stanza += QString::number(d, 'f', precision_spec);
        }
      else if(integerKeys.contains(key))
        {
          // ---- INTEGER TOKENS ----
          qlonglong i = 0;

          // INTEGER tokens should accept:
          // int, uint, qint64, quint64        (ideal case)
          // double or float                    (we can convert by
          // truncation) string with digits                 (allowed at your
          // discretion)

          if(v.canConvert<qlonglong>())
            i = v.toLongLong();
          else
            {
              stanza    += "<NaI>"; // Not-an-Integer marker (or use literal)
              lastIndex  = end;
              continue;
            }

          stanza += QString::number(i);
        }

      else if(stringKeys.contains(key))
        {
          // ---- STRING TOKENS ----
          if(v.canConvert<QString>())
            stanza += v.toString();
          else
            stanza += "<NaS>"; // Not-a-String
        }

      lastIndex = end;
    }

  // trailing literal data
  stanza += formatString.mid(lastIndex);

  // Finally, set the stanza to the member datum and return it.
  // We want to keep it as a member datum, because derived classes
  // can build upon it with specialized data elements.
  m_analysisStanza = stanza;

  // qDebug() << "Returning m_analysisStanza:" << m_analysisStanza;

  // Note that each time a stanza has been crafted, we need to reset the tic
  // intensity value to 0 so that we do not craft two stanzas with the same
  // value if no single intensity calculation was performed between them.

  return m_analysisStanza;
}

void
MassSpecTracePlotCompositeWidget::
  setMinimalChargeFractionalPartPinnedDownWidgets(double value)
{
  static_cast<pappso::MassSpecTracePlotWidget *>(mp_plotWidget)
    ->setChargeMinimalFractionalPart(value);
}

void
MassSpecTracePlotCompositeWidget::
  setChargeStateEnvelopePeakSpanPinnedDownWidgets(std::size_t value)
{
  static_cast<pappso::MassSpecTracePlotWidget *>(mp_plotWidget)
    ->setChargeStateEnvelopePeakSpan(value);
}


} // namespace MineXpert

} // namespace MsXpS
